/*******************************************************************************
 * Copyright (c) 2012, 2020 IBM Corp.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * and Eclipse Distribution License v1.0 which accompany this distribution.
 *
 * The Eclipse Public License is available at
 *   https://www.eclipse.org/legal/epl-2.0/
 * and the Eclipse Distribution License is available at
 *   http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *    Ian Craggs - initial contribution
 *******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "MQTTAsync.h"
#include "mqtt_config.h"

static MQTTAsync s_Client;
int finished = 0;

void connlost(void *context, char *cause)
{
	MQTTAsync client = (MQTTAsync)context;
	MQTTAsync_connectOptions conn_opts = MQTTAsync_connectOptions_initializer;
	int rc;

	printf("\nConnection lost\n");
	printf("     cause: %s\n", cause);

	printf("Reconnecting\n");
	conn_opts.keepAliveInterval = 20;
	conn_opts.cleansession = 1;
	if ((rc = MQTTAsync_connect(client, &conn_opts)) != MQTTASYNC_SUCCESS)
	{
		printf("Failed to start connect, return code %d\n", rc);
		finished = 1;
	}
}

void onDisconnectFailure(void* context, MQTTAsync_failureData* response)
{
	printf("Disconnect failed\n");
	finished = 1;
}

void onDisconnect(void* context, MQTTAsync_successData* response)
{
	printf("Successful disconnection\n");
	finished = 1;
}

void onSendFailure(void* context, MQTTAsync_failureData* response)
{
	MQTTAsync client = (MQTTAsync)context;
	MQTTAsync_disconnectOptions opts = MQTTAsync_disconnectOptions_initializer;
	int rc;

	printf("Message send failed token %d error code %d\n", response->token, response->code);
	opts.onSuccess = onDisconnect;
	opts.onFailure = onDisconnectFailure;
	opts.context = client;
	if ((rc = MQTTAsync_disconnect(client, &opts)) != MQTTASYNC_SUCCESS)
	{
		printf("Failed to start disconnect, return code %d\n", rc);
	}
}

void onSend(void* context, MQTTAsync_successData* response)
{
	MQTTAsync client = (MQTTAsync)context;
	MQTTAsync_disconnectOptions opts = MQTTAsync_disconnectOptions_initializer;
	int rc;

	printf("Message with token value %d delivery confirmed\n", response->token);
}


void onConnectFailure(void* context, MQTTAsync_failureData* response)
{
	printf("Connect failed, rc %d\n", response ? response->code : 0);
	finished = 1;
}


void onConnect(void* context, MQTTAsync_successData* response)
{
	MQTTAsync client = (MQTTAsync)context;
	MQTTAsync_responseOptions opts = MQTTAsync_responseOptions_initializer;
	MQTTAsync_message pubmsg = MQTTAsync_message_initializer;
	int rc;

	printf("Successful connection\n");
	opts.onSuccess = onSend;
	opts.onFailure = onSendFailure;
	opts.context = client;
	pubmsg.payload = PAYLOAD;
	pubmsg.payloadlen = (int)strlen(PAYLOAD);
	pubmsg.qos = QOS;
	pubmsg.retained = 0;
	if ((rc = MQTTAsync_sendMessage(client, TOPIC, &pubmsg, &opts)) != MQTTASYNC_SUCCESS)
	{
		printf("Failed to start sendMessage, return code %d\n", rc);
	}
}


int messageArrived(void* context, char* topicName, int topicLen, MQTTAsync_message* message)
{
	/* not expecting any messages */
	printf("onMessageArrived topic:%s,message length:%d.\n",topicName,message->payloadlen);
	printf("payload:%s,\n",message->payload);
	KK_MQTT_RecvMsg((MQTTAsync)context,topicName,message->payload);
	MQTTAsync_freeMessage(&message);
	MQTTAsync_free(topicName);
	return 1;
}

static void mqttTraceCallback(enum MQTTASYNC_TRACE_LEVELS level, char *message)
{
	//printf("mqttTraceCallback level:%d,msg:%s.\n",level,message);
}
static void onDeliveryComplete(void* context, MQTTAsync_token token)
{
	printf("onDeliveryComplete,token:%d \n",token);
}
static void onConnectBuild(void *context, char *cause)
{
	printf("onConnectBuild:%s \n",cause);
	KK_MQTT_SubTopic(s_Client,TOPIC,0,2000);
}
static void onDisConnected(void *context, MQTTProperties* properties,enum MQTTReasonCodes reasonCode)
{
	printf("onDisConnected,maybe kicked by broker.\n");
}

static void mqtt_set_callbacks(void)
{
	MQTTAsync_setConnectionLostCallback(s_Client,NULL,connlost);
	MQTTAsync_setMessageArrivedCallback(s_Client,NULL,messageArrived);
	MQTTAsync_setDeliveryCompleteCallback(s_Client,NULL,onDeliveryComplete);
	MQTTAsync_setConnected(s_Client,NULL,onConnectBuild);
	MQTTAsync_setDisconnected(s_Client,NULL,onDisConnected);

}
static int mqtt_start(void)
{
	int count = 0;
	MQTTAsync_connectOptions conn_opts = MQTTAsync_connectOptions_initializer;
	int rc = 0;
	MQTTAsync_createOptions opts  = MQTTAsync_createOptions_initializer;

	MQTTAsync_setTraceCallback(mqttTraceCallback);

	if ((rc = MQTTAsync_createWithOptions(&s_Client, ADDRESS, CLIENTID, MQTTCLIENT_PERSISTENCE_NONE, NULL,&opts)) != MQTTASYNC_SUCCESS)
	{
		printf("Failed to create client object, return code %d\n", rc);
		return -1;
	}
	mqtt_set_callbacks();
	opts.MQTTVersion = MQTTVERSION_3_1_1;	
	conn_opts.keepAliveInterval = 60;
	conn_opts.connectTimeout = CONNECT_TIMEOUT;
	conn_opts.automaticReconnect = AUTO_CONN;
	conn_opts.minRetryInterval = 1;
	conn_opts.maxRetryInterval = 32;
	conn_opts.username = USRNAME;
	conn_opts.password = PASSWORD;
	conn_opts.keepAliveInterval = 20;
	conn_opts.cleansession = 1;
	conn_opts.onSuccess = onConnect;
	conn_opts.onFailure = onConnectFailure;
	conn_opts.context = s_Client;	
	if ((rc = MQTTAsync_connect(s_Client, &conn_opts)) != MQTTASYNC_SUCCESS)
	{
		printf("Failed to start connect, return code %d\n", rc);
		return -1;
	}
	printf("Waiting for publication of %s\n"
	       "on topic %s for client with ClientID: %s\n",
	       PAYLOAD, TOPIC, CLIENTID);
	while (!finished)
	{
		usleep(100000L);
		
		count++;
		if(count>50)
		{
			count = 0;
			printf("i am alive now\n");
			KK_MQTT_SendMsg(s_Client,TOPIC,"hello my world",2);
		}
		
	}
	printf("MQTTAsync_destroy\n");
	MQTTAsync_destroy(&s_Client);
	return rc;

}



int main(int argc, char* argv[])
{
	int rc = 0;
	rc = mqtt_start();
	return rc;
}

