/*
  Copyright (c) 2009 Dave Gamble
 
  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:
 
  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.
 
  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/

#ifndef rpccJSON__h
#define rpccJSON__h

#ifdef __cplusplus
extern "C"
{
#endif

/* cJSON Types: */
#define cJSON_False 0
#define cJSON_True 1
#define cJSON_NULL 2
#define cJSON_Number 3
#define cJSON_String 4
#define cJSON_Array 5
#define cJSON_Object 6
	
#define cJSON_IsReference 256

/* The cJSON structure: */
typedef struct cJSON {
	struct cJSON *next,*prev;	/* next/prev allow you to walk array/object chains. Alternatively, use GetArraySize/GetArrayItem/GetObjectItem */
	struct cJSON *child;		/* An array or object item will have a child pointer pointing to a chain of the items in the array/object. */

	int type;					/* The type of the item, as above. */

	char *valuestring;			/* The item's string, if type==cJSON_String */
	int valueint;				/* The item's number, if type==cJSON_Number */
	double valuedouble;			/* The item's number, if type==cJSON_Number */

	char *string;				/* The item's name string, if this item is the child of, or is in the list of subitems of an object. */
} cJSON;

typedef struct cJSON_Hooks {
      void *(*malloc_fn)(size_t sz);
      void (*free_fn)(void *ptr);
} cJSON_Hooks;

/* Supply malloc, realloc and free functions to cJSON */
extern void rpc_cJSON_InitHooks(cJSON_Hooks* hooks);


/* Supply a block of JSON, and this returns a cJSON object you can interrogate. Call rpc_cJSON_Delete when finished. */
extern cJSON *rpc_cJSON_Parse(const char *value);
/* Supply a block of JSON, and this returns a cJSON object you can interrogate. Call rpc_cJSON_Delete when finished.
 * end_ptr will point to 1 past the end of the JSON object */
extern cJSON *rpc_cJSON_Parse_Stream(const char *value, char **end_ptr);
/* Render a cJSON entity to text for transfer/storage. Free the char* when finished. */
extern char  *rpc_cJSON_Print(cJSON *item);
/* Render a cJSON entity to text for transfer/storage without any formatting. Free the char* when finished. */
extern char  *rpc_cJSON_PrintUnformatted(cJSON *item);
/* Delete a cJSON entity and all subentities. */
extern void   rpc_cJSON_Delete(cJSON *c);

/* Returns the number of items in an array (or object). */
extern int	  rpc_cJSON_GetArraySize(cJSON *array);
/* Retrieve item number "item" from array "array". Returns NULL if unsuccessful. */
extern cJSON *rpc_cJSON_GetArrayItem(cJSON *array,int item);
/* Get item "string" from object. Case insensitive. */
extern cJSON *rpc_cJSON_GetObjectItem(cJSON *object,const char *string);

/* These calls create a cJSON item of the appropriate type. */
extern cJSON *rpc_cJSON_CreateNull();
extern cJSON *rpc_cJSON_CreateTrue();
extern cJSON *rpc_cJSON_CreateFalse();
extern cJSON *rpc_cJSON_CreateBool(int b);
extern cJSON *rpc_cJSON_CreateNumber(double num);
extern cJSON *rpc_cJSON_CreateString(const char *string);
extern cJSON *rpc_cJSON_CreateArray();
extern cJSON *rpc_cJSON_CreateObject();

/* These utilities create an Array of count items. */
extern cJSON *rpc_cJSON_CreateIntArray(int *numbers,int count);
extern cJSON *rpc_cJSON_CreateFloatArray(float *numbers,int count);
extern cJSON *rpc_cJSON_CreateDoubleArray(double *numbers,int count);
extern cJSON *rpc_cJSON_CreateStringArray(const char **strings,int count);

/* Append item to the specified array/object. */
extern void rpc_cJSON_AddItemToArray(cJSON *array, cJSON *item);
extern void	rpc_cJSON_AddItemToObject(cJSON *object,const char *string,cJSON *item);
/* Append reference to item to the specified array/object. Use this when you want to add an existing cJSON to a new cJSON, but don't want to corrupt your existing cJSON. */
extern void rpc_cJSON_AddItemReferenceToArray(cJSON *array, cJSON *item);
extern void	rpc_cJSON_AddItemReferenceToObject(cJSON *object,const char *string,cJSON *item);

/* Remove/Detatch items from Arrays/Objects. */
extern cJSON *rpc_cJSON_DetachItemFromArray(cJSON *array,int which);
extern void   rpc_cJSON_DeleteItemFromArray(cJSON *array,int which);
extern cJSON *rpc_cJSON_DetachItemFromObject(cJSON *object,const char *string);
extern void   rpc_cJSON_DeleteItemFromObject(cJSON *object,const char *string);
	
/* Update array items. */
extern void rpc_cJSON_ReplaceItemInArray(cJSON *array,int which,cJSON *newitem);
extern void rpc_cJSON_ReplaceItemInObject(cJSON *object,const char *string,cJSON *newitem);

#define rpc_cJSON_AddNullToObject(object,name)	rpc_cJSON_AddItemToObject(object, name, rpc_cJSON_CreateNull())
#define rpc_cJSON_AddTrueToObject(object,name)	rpc_cJSON_AddItemToObject(object, name, rpc_cJSON_CreateTrue())
#define rpc_cJSON_AddFalseToObject(object,name)		rpc_cJSON_AddItemToObject(object, name, rpc_cJSON_CreateFalse())
#define rpc_cJSON_AddNumberToObject(object,name,n)	rpc_cJSON_AddItemToObject(object, name, rpc_cJSON_CreateNumber(n))
#define rpc_cJSON_AddStringToObject(object,name,s)	rpc_cJSON_AddItemToObject(object, name, rpc_cJSON_CreateString(s))

#ifdef __cplusplus
}
#endif

#endif
